<?php

namespace App\Http\Controllers;

use App\Http\Controllers\CompanyFeatureUsageController;
use App\Models\Job;
use App\Events\JobPosted;
use App\Models\CompanyFeatureUsage; // Import the model
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class JobController extends Controller
{
    public function index(): JsonResponse
    {
        $jobs = Job::with(['company.user'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $jobs,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $job = Job::with(['company.user'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $job,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Company_id' => 'required|exists:company_profiles,Company_id',
            'Title' => 'required|string|max:255',
            'Description' => 'required|string',
            'Location' => 'nullable|string|max:255',
            'Salary' => 'nullable|numeric|min:0',
            'Job_type' => 'required|in:full_time,part_time,contract,internship',
            'Status' => 'sometimes|in:pending,approved,rejected,closed',
            'deadline' => 'nullable|date|after:' . now()->format('Y-m-d H:i:s'),
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Get or create the usage record
        $companyId = $request->input('Company_id');
        $featureName = 'job_postings';
        $usage = CompanyFeatureUsage::firstOrCreate(
            [
                'Company_id' => $companyId,
                'Feature_Name' => $featureName,
            ],
            [
                'Plan_id' => null, // Set default or fetch from subscription
                'Subscription_id' => null, // Set default or fetch from company profile
                'Usage_Count' => 0,
                'Max_Usage' => 10, // Default max usage, adjust based on plan
                'Start_Date' => now(),
                'End_Date' => null,
            ]
        );

        // Check and increment job posting usage
        try {
            $usageController = new CompanyFeatureUsageController();
            $usageResponse = $usageController->incrementUsage(new Request([
                'Company_id' => $companyId,
                'Feature_Name' => $featureName,
            ]));

            if ($usageResponse->getStatusCode() !== 200) {
                return $usageResponse;
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to check usage limit: ' . $e->getMessage(),
            ], 500);
        }

        $job = Job::create([
            'Company_id' => $request->input('Company_id'),
            'Title' => $request->input('Title'),
            'Description' => $request->input('Description'),
            'Location' => $request->input('Location'),
            'Salary' => $request->input('Salary'),
            'Job_type' => $request->input('Job_type'),
            'Status' => 'pending',
            'deadline' => $request->input('deadline'),
        ]);

        // Trigger the JobPosted event for notifications
        event(new JobPosted($job));

        return response()->json([
            'status' => 'success',
            'data' => $job,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $job = Job::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Company_id' => 'sometimes|exists:company_profiles,Company_id',
            'Title' => 'sometimes|string|max:255',
            'Description' => 'sometimes|string',
            'Location' => 'nullable|string|max:255',
            'Salary' => 'nullable|numeric|min:0',
            'Job_type' => 'sometimes|in:full_time,part_time,contract,internship',
            'Status' => 'sometimes|in:pending,approved,rejected,closed',
            'deadline' => 'nullable|date|after:' . now()->format('Y-m-d H:i:s'),
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $job->update($request->only([
            'Company_id',
            'Title',
            'Description',
            'Location',
            'Salary',
            'Job_type',
            'Status',
            'deadline',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $job,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $job = Job::findOrFail($id);
        $job->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Job deleted successfully',
        ], 200);
    }

    public function jobCounts(): JsonResponse
    {
        $totalJobs = Job::count();
        $pendingJobs = Job::where('Status', 'pending')->count();
        $approvedJobs = Job::where('Status', 'approved')->count();

        return response()->json([
            'status' => 'success',
            'data' => [
                'total_jobs' => $totalJobs,
                'pending_jobs' => $pendingJobs,
                'approved_jobs' => $approvedJobs,
            ],
        ], 200);
    }
}
